﻿
CREATE PROCEDURE [App].[NVCC_usp_UsageData]
	-- @timegroup is a character parameter which can take values NULL, 'continuous',
	-- 'day', 'week', 'month', or 'ever'. NULL is equivalent to 'continuous'. Any other value
	-- will result in a 0-row dataset.
	-- For 'continuous' (or NULL), the columns AccessDateTime and Ord will be populated
	-- with Ord giving the ordering (rank) within the specified groups.
	-- For 'day', the columns Date and N will be populated where N is the number of
	-- reports for each Date, further split by any specified groups.
	-- For 'week', the columns Week and N will be populated where N is the number of
	-- reports for each Week, further split by any specified groups.
	-- For 'month', the columns YearMon and N will be populated where N is the number of
	-- reports for each YearMon, further split by any specified groups.
	-- For 'ever', the column N will be populated where N is the number of reports,
	-- further split by any specified groups.
	@timegroup varchar(max) NULL,
	-- @sta3n is a character parameter which can either NULL, a single station number,
	-- or a comma demlimited list of station numbers. NULL implies that the data for
	-- all stations should be returned/used and that no stratification by station should
	-- be done. A single station number (as a string) will return data for only that
	-- station. A comma delimited list of station numbers will restrict the returned data
	-- to those present in the list and any aggregation or ordination will be done
	-- separately for each station.
	@sta3n varchar(max) NULL,
	-- @userid is a character parameter which can either NULL, a single user identifier,
	-- or a comma demlimited list of user identifiers. A user identifier is a login,
	-- including the domain. NULL implies that the data for all userss stations should
	-- be returned/used and that no stratification by user should be done. A userid
	-- will return data for only that user. A comma delimited list of users will
	-- restrict the returned data to those present in the list and any aggregation
	-- or ordination will be done separately for each user.
	@userid varchar(max) NULL
AS
--==================================================================================
--Requestor             : Non Va Coordinated Care Project
--Author                : Brian Diggs
--Object/SP Name        : NVCC_usp_GetLogs
--Server                : 
--Data Base             : CC_REFDOC
--Schema                : app
--Report                : N/A
--Folder                : N/A
--Report Loc            : N/A
--Job                   : None
--Job Loc               : None
--Note                  : 
--Date Created          : 05-13-2016
--
--
--Last Changed          : 
--Last Changed By		: 
--Reason For Change		: 
--
--Purpose               : To get aggregated, subsetted production log data.
--==================================================================================
--Uses: App.udf_SplitString
--Uses: App.NVCC_AccessLog
--Uses: App.NVCC_Users
--Uses: CDWWork.Patient.Patient
--Uses: CDWWork.SStaff.SStaff

BEGIN
	-- First, create a temporary table, #LogData, which pulls records from
	-- App.NVCC_AccessLog which are:
	--     On the production server
	--     PDF generations
	--     Not by the internal NWIC team
	--
	-- The AccessDateTime and UserID are kept from the log, and the
	-- PatientSID is used to determine the Sta3n for that patient.
	--
	-- This list is further restricted to those stations and users specified
	-- by the parameters. This is accomplished by converting the parameter
	-- string to a table using App.udf_SplitString. Only the value is kept
	-- and, for the station, converted to a smallint to match the datatype
	-- elsewhere. An input of NULL will give a single row table with value
	-- NULL. These conversions are done a common table expressions.
	-- These table are INNER JOINed to the data, with the clause that
	-- everything is kept of the value is NULL.
	--
	-- UserID and Sta3n are converted to NULL is their respective parameters
	-- are NULL. This indicates that there is no stratification done (or
	-- possible) on those variables. Derived time quantities are created
	-- from the AccessDateTime giving the appropriate day, week (Sunday
	-- through Saturday), and year/month.
	WITH
		Sta3n AS (SELECT CONVERT(smallint, Value) AS Sta3n FROM App.udf_SplitString(@sta3n, ',')),
		UserID AS (SELECT Value AS UserID FROM App.udf_SplitString(@userid, ','))
		SELECT
			al.AccessDateTime,
			IIF(@userid IS NULL, NULL, al.UserID) AS UserID,
			IIF(@sta3n IS NULL, NULL, p.Sta3n) AS Sta3n,
			CONVERT(DATE, AccessDateTime) AS Date,
			CONVERT(VARCHAR, CONVERT(DATE, DATEADD(DAY, 7 * (DATEDIFF(DAY, '1990-01-07', al.AccessDateTime) / 7), '1990-01-07'))) + 
				' - ' + CONVERT(VARCHAR, CONVERT(DATE, DATEADD(DAY, (7 * (DATEDIFF(DAY, '1990-01-07', al.AccessDateTime) / 7)) + 6, '1990-01-07'))) AS Week,
			FORMAT(DATEPART(YEAR, al.AccessDateTime), '0000') + '-' + FORMAT(DATEPART(MONTH, al.AccessDateTime), '00') AS YearMon
		INTO #LogData
		FROM
			App.vNVCC_ProductionAccessLog al
			INNER JOIN CDWWork.Patient.Patient p
				ON al.PatientSID = p.PatientSID
			INNER JOIN Sta3n st
				ON (p.Sta3n = st.Sta3n OR st.Sta3n IS NULL)
			INNER JOIN UserID uid
				ON (al.UserID = uid.UserID OR uid.UserID IS NULL)

	-- A temporary table, #LogAgg, is created based on the structure of #LogAgg plus two
	-- other, NULLable numeric columns Ord and N. This must be done in a separate step
	-- step (rather than implicitly with at SELECT ... INTO) because the same named
	-- temporary table can not be used as a target for multiple SELECT ... INTO statements.
	-- The WHERE clause is designed to select no rows, so no data is initially seeded
	-- in the temporary table.
	SELECT
		ld.*,
		CONVERT(integer, NULL) AS Ord,
		CONVERT(integer, NULL) AS N
	INTO #LogAgg
	FROM #LogData AS ld
	WHERE 1=0;

	-- Choose the code path based on the value of @timegroup.
	IF (@timegroup = 'continuous' or @timegroup is null)
		BEGIN
			INSERT INTO #LogAgg (Sta3n, UserID, AccessDateTime, Ord)
			SELECT
				Sta3n,
				UserID,
				AccessDateTime,
				ROW_NUMBER() OVER (PARTITION BY UserID, Sta3n ORDER BY AccessDateTime ASC) AS Ord
			FROM #LogData
			ORDER BY AccessDateTime
		END
	ELSE IF (@timegroup = 'day')
		BEGIN
			INSERT INTO #LogAgg (Date, Sta3n, UserID, N)
			SELECT
				Date,
				Sta3n,
				UserID,
				COUNT(*) AS N
			FROM #LogData
			GROUP BY
				Date, Sta3n, UserID
			ORDER BY
				DATE
		END
	ELSE IF (@timegroup = 'week')
		BEGIN
			INSERT INTO #LogAgg (Week, Sta3n, UserID, N)
			SELECT
				Week,
				Sta3n,
				UserID,
				COUNT(*) AS N
			FROM #LogData
			GROUP BY
				Week, Sta3n, UserID
			ORDER BY
				WEEK
		END
	ELSE IF (@timegroup = 'month')
		BEGIN
			INSERT INTO #LogAgg (YearMon, Sta3n, UserID, N)
			SELECT
				YearMon,
				Sta3n,
				UserID,
				COUNT(*) AS N
			FROM #LogData
			GROUP BY
				YearMon, Sta3n, UserID
			ORDER BY
				YearMon
		END
	ELSE IF (@timegroup = 'ever')
		BEGIN
			INSERT INTO #LogAgg (Sta3n, UserID, N)
			SELECT
				Sta3n,
				UserID,
				COUNT(*) AS N
			FROM #LogData
			GROUP BY
				Sta3n, UserID
		END

	-- This temporary table is no longer needed.
	DROP TABLE #LogData;

	-- Further annotate the #LogAgg table by converting the UserID
	-- into the Staff Name and converting the station number into
	-- a more usable version of the station name based on the
	-- App.udf_StationName function.
	--
	-- An additional ID column is included so that Entity Framework
	-- has something to use as a primary key. Ironically, there is
	-- no column in the table which is guaranteed to be non-NULL
	-- for all values of the parameters. And apparently, having
	-- ID as the ORDER BY inside the OVER clause defining ID
	-- isn't invalid. I have no idea what it means, though.
	-- However, since the ID is not semantically meaningful (it
	-- just has to be unique), it doesn't matter.
	--
	-- Return this further annotated table.
	SELECT
		ROW_NUMBER() OVER (ORDER BY ID) AS ID,
		la.*,
		s.StaffName,
		App.udf_StationName(la.Sta3n) AS Station
	FROM
		#LogAgg AS la
		LEFT JOIN App.NVCC_Users u
			ON la.UserID = u.DomainPlusNetworkUserName
		LEFT JOIN CDWWork.SStaff.SStaff s
			ON u.UserIEN = s.StaffIEN and u.Sta3n = s.Sta3n
END

